Imports MicroFour.StrataFrame.Business
Imports MicroFour.StrataFrame.UI.Windows.Forms
Imports MicroFour.StrataFrame.Extensibility
Imports System.ComponentModel
Imports System.Drawing.Design
Imports System.Windows.Forms
Imports MicroFour.StrataFrame.Tools
Imports LookUpControlSample_BO


Public Class GenericLookUpControl(Of TBrowseDialog As {BrowseDialog, New})
  Implements IBusinessBindable


#Region " Constructors "

  Public Sub New()
    Me.InitializeComponent()

    '-- Init browse dialog and business object.
    Me.Init()

    '-- Bind the code to the textbox
    Me.txtCode.DataBindings.Add("Text", Me, "ID", True, DataSourceUpdateMode.OnPropertyChanged)

  End Sub

#End Region

#Region " Events "

  ''' <summary>
  ''' Occurs when the return value of the ID property changes.
  ''' </summary>
  ''' <remarks></remarks>
  Public Event IDChanged As EventHandler

  ''' <summary>
  ''' Raises the IDChanged event.
  ''' </summary>
  ''' <remarks></remarks>
  Protected Overridable Sub OnIDChanged()
    RaiseEvent IDChanged(Me, EventArgs.Empty)
  End Sub

#End Region

#Region " Handled Events "

  Private Sub cmdBrowse_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles cmdBrowse.Click
    '-- Show the browse dialog
    If Me.LookupDialog.ShowDialog() = DialogResult.OK Then
      '-- Set ID to PK of current record in lookup BO.
      _LoadedBySearch = True
      Me.ID = CInt(Me.LookpBusinessObject(Me.LookpBusinessObject.PrimaryKeyField))
    End If
  End Sub

#End Region

#Region " IBusinessBindable Implementation "

  Private _BusinessObject As BusinessLayer
  Private _BindingField As String = ""
  Private _BindingFormat As String = ""
  Private _BindingProperty As String = "ID"
  Private _IgnoreManageUIReadOnlyState As Boolean = True
  Private _BindingUpdateMode As DataSourceUpdateMode = DataSourceUpdateMode.OnPropertyChanged
  Private _ControlSourceUpdateMode As ControlUpdateMode = ControlUpdateMode.OnPropertyChanged

  ''' <summary>
  ''' The field within the business object that is bound to the control
  ''' </summary>
  <Category(EDITOR_CATEGORY), Description(EDITOR_BINDINGUPDATEMODE_DESC)> _
  Public Property BindingUpdateMode() As DataSourceUpdateMode Implements IBusinessBindable.DataSourceUpdateMode
    Get
      Return _BindingUpdateMode
    End Get
    Set(ByVal value As System.Windows.Forms.DataSourceUpdateMode)
      _BindingUpdateMode = value
    End Set
  End Property

  ''' <summary>
  ''' The field within the business object that is bound to the control
  ''' </summary>
  ''' <value></value>
  ''' <remarks></remarks>
  <Category(EDITOR_CATEGORY), _
  DefaultValue(""), _
  Description(EDITOR_BINDINGFIELD_DESC), _
  Editor(MicroFour.StrataFrame.Extensibility.Constants.TE_BindingFieldEditor, GetType(UITypeEditor))> _
  Public Property BindingField() As String Implements IBusinessBindable.BindingField
    Get
      Return _BindingField
    End Get
    Set(ByVal value As String)
      _BindingField = value
    End Set
  End Property

  ''' <summary>
  ''' The business object that is used to bind the data to this field
  ''' </summary>
  ''' <value></value>
  ''' <remarks></remarks>
  <Category(EDITOR_CATEGORY), _
  DefaultValue(CType(Nothing, Object)), _
  Description(EDITOR_BUSINESSOBJECT_DESC)> _
  Public Property BusinessObject() As BusinessLayerBase Implements IBusinessBindable.BusinessObject
    Get
      Return Me._BusinessObject
    End Get
    Set(ByVal value As BusinessLayerBase)
      '-- If the business object is being removed, then make sure it is no longer bound
      If (value Is Nothing) AndAlso (_BusinessObject IsNot Nothing) Then
        Me._BusinessObject.RemoveFromListOfBoundControls(Me)
      End If

      If (Not Me.DesignMode) AndAlso (value IsNot Nothing) Then
        value.AddToListOfBoundControls(Me)
      End If

      _BusinessObject = CType(value, BusinessLayer)
    End Set
  End Property

  ''' <summary>
  ''' The property on this control to which the data is bound.
  ''' </summary>
  ''' <value></value>
  ''' <remarks></remarks>
  <Category(EDITOR_CATEGORY), _
  DefaultValue("ID"), _
  Description(EDITOR_BINDINGPROPERTY_DESC), _
  Editor(MicroFour.StrataFrame.Extensibility.Constants.TE_BindingPropertyEditor, GetType(UITypeEditor))> _
  Public Property BindingProperty() As String Implements IBusinessBindable.BindingProperty
    Get
      Return _BindingProperty
    End Get
    Set(ByVal value As String)
      _BindingProperty = value
    End Set
  End Property

  ''' <summary>
  ''' If True, the control will ignore that auto manage editing state of the business object.  The control will not
  ''' be automatically enabled/disabled by the business object.
  ''' </summary>
  ''' <value></value>
  ''' <remarks></remarks>
  <Category(EDITOR_CATEGORY), _
  DefaultValue(True), _
  Description(EDITOR_IGNOREMANAGE_DESC)> _
  Public Property IgnoreManageReadOnlyState() As Boolean Implements IBusinessBindable.IgnoreManageUIReadOnlyState
    Get
      Return _IgnoreManageUIReadOnlyState
    End Get
    Set(ByVal value As Boolean)
      _IgnoreManageUIReadOnlyState = value
    End Set
  End Property

  ''' <summary>
  ''' The formst string used to format the data that is bound to the control
  ''' </summary>
  ''' <value></value>
  ''' <remarks></remarks>
  <Description(EDITOR_BINDINGFORMAT_DESC), Category(EDITOR_CATEGORY), DefaultValue("")> _
  Public Property BindingFormat() As String Implements IBusinessBindable.BindingFormat
    Get
      Return _BindingFormat
    End Get
    Set(ByVal value As String)
      _BindingFormat = value
    End Set
  End Property

  ''' <summary>
  ''' Makes the control available or unavailable for editing
  ''' </summary>
  ''' <value></value>
  ''' <remarks></remarks>
  <Browsable(False)> _
  Public Property BindingEditable() As Boolean Implements IBusinessBindable.BindingEditable
    Get
      Return Me.Enabled
    End Get
    Set(ByVal value As Boolean)
      Me.Enabled = value
    End Set
  End Property

  ''' <summary>
  ''' Gets or sets a value that determines whether the control will allow
  ''' its editable state to be managed by the bound business object.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  <Category(EDITOR_CATEGORY), _
  Description(EDITOR_CONTROLUPDATEMODE_DESC)> _
  Public Property ControlSourceUpdateMode() As System.Windows.Forms.ControlUpdateMode Implements IBusinessBindable.ControlSourceUpdateMode
    Get
      Return Me._ControlSourceUpdateMode
    End Get
    Set(ByVal value As System.Windows.Forms.ControlUpdateMode)
      Me._ControlSourceUpdateMode = value
    End Set
  End Property

  Private Sub ResetControlSourceUpdateMode()
    Me._ControlSourceUpdateMode = ControlUpdateMode.OnPropertyChanged
  End Sub

  Private Function ShouldSerializeControlSourceUpdateMode() As Boolean
    Return Me._ControlSourceUpdateMode <> ControlUpdateMode.OnPropertyChanged
  End Function

  ''' <summary>
  ''' Determines if the binding update mode property is set to default
  ''' </summary>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Private Function ShouldSerializeBindingUpdateMode() As Boolean Implements IBusinessBindable.ShouldSerializeBindingUpdateMode
    Return _BindingUpdateMode <> DataSourceUpdateMode.OnPropertyChanged
  End Function

  ''' <summary>
  ''' Resets the binding update mode
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub ResetBindingUpdateMode() Implements IBusinessBindable.ResetBindingUpdateMode
    _BindingUpdateMode = DataSourceUpdateMode.OnPropertyChanged
  End Sub

#End Region

#Region " Private Fields "

  Private _LoadedBySearch As Boolean = False
  Private _ID As Integer = 0
  Private _DescriptionTemplate As String = "{0}"
  Private _DescriptionFieldNames As String()
  Private _LookupBusinessObject As _BaseBO
  Private _LookupDialog As BrowseDialog

#End Region

#Region " Private Methods "

  ''' <summary>
  ''' Get the description for the selected item.
  ''' </summary>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Private Function GetDescription()
    Dim desc As String = String.Empty

    '-- Create an array of the data needed.
    If Me.LookpBusinessObject IsNot Nothing AndAlso Me.LookpBusinessObject.Count > 0 Then
      Dim data As New ArrayList()
      For Each field As String In Me.DescriptionFieldNames
        data.Add(Me.LookpBusinessObject(field))
      Next
      desc = String.Format(Me.DescriptionTemplate, data.ToArray())
    End If

    Return desc
  End Function

  ''' <summary>
  ''' Initialize the browse dialog and the business object.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub Init()
    '-- Create dialog
    _LookupDialog = Activator.CreateInstance(GetType(TBrowseDialog))

    '-- Create and setup the Business object.
    Dim boTypeName As String = _LookupDialog.BusinessObjectType
    _LookupBusinessObject = CType(Activator.CreateInstance(Common.GetTypeFromReferencedAssemblies(boTypeName)), _BaseBO)
    _LookupBusinessObject.ParentContainer = Me
    _LookupBusinessObject.SynchronizingObject = Me

    '-- Setup dialog
    _LookupDialog.BusinessObjectToPopulate = _LookupBusinessObject

  End Sub

  ''' <summary>
  ''' Loads the data into the control for the indicated PK.
  ''' </summary>
  ''' <remarks></remarks>
  Private Sub LoadData(ByVal dataPk As Integer)
    '-- Skip if already loaded by the search.
    If _LoadedBySearch Then
      _LoadedBySearch = False
      Return
    End If

    '-- Establish Locals
    Dim dataCode As String = ""

    '-- Load BO with data for this pk
    Me.LookpBusinessObject.FillByPrimaryKey(dataPk)

    '-- Set the text
    Me.lblDesc.Text = Me.GetDescription()
  End Sub

#End Region

#Region " Public Properties "

  ''' <summary>
  ''' Setup search fields for browse dialog.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Public ReadOnly Property LookupDialog() As BrowseDialog
    Get
      Return _LookupDialog
    End Get
  End Property

  ''' <summary>
  ''' Business object of specified type.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  Public ReadOnly Property LookpBusinessObject() As _BaseBO
    Get
      Return _LookupBusinessObject
    End Get
  End Property

  ''' <summary>
  ''' Define the template that is used to build the description.
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  <Category("Lookup Description Setup")> _
  <Description("Define the template used to build the description. This is a String.Format template. It uses the field names defined in the DescriptionFieldNames property.")> _
  <DefaultValue("{0}")> _
  Public Property DescriptionTemplate() As String
    Get
      Return _DescriptionTemplate
    End Get
    Set(ByVal value As String)
      _DescriptionTemplate = value
    End Set
  End Property

  ''' <summary>
  ''' Define the field names used to build the description. 
  ''' </summary>
  ''' <value></value>
  ''' <returns></returns>
  ''' <remarks></remarks>
  <Category("Lookup Description Setup")> _
  <Description("Define the field names used to build the description.")> _
  Public Property DescriptionFieldNames() As String()
    Get
      Return _DescriptionFieldNames
    End Get
    Set(ByVal value As String())
      _DescriptionFieldNames = value
    End Set
  End Property

  ''' <summary>
  ''' Define the ID of the loaded record.
  ''' </summary>
  <Browsable(False)> _
  <DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)> _
  Public Property ID() As Integer
    Get
      Return Me._ID
    End Get
    Set(ByVal value As Integer)
      If value <> _ID Then
        Me._ID = value

        '-- Load the display area as configured.
        Me.LoadData(_ID)
        Me.lblDesc.Text = Me.GetDescription()

        OnIDChanged()
      End If
    End Set
  End Property

  ''' <summary>
  ''' Return if the designer should serialize the 
  ''' DescriptionFieldNames property.
  ''' </summary>
  ''' <returns>True if property needs to be serialized to the designer</returns>
  ''' <remarks></remarks>     
  Public Function ShouldSerializeDescriptionFieldNames() As Boolean
    Return _DescriptionFieldNames IsNot Nothing
  End Function

  ''' <summary>
  ''' Reset the DescriptionFieldNames to its default value and remove serialization
  ''' the property from the designer.
  ''' </summary>
  ''' <remarks></remarks>     
  Public Sub ResetDescriptionFieldNames()
    _DescriptionFieldNames = Nothing
  End Sub


#End Region

End Class
